/*
 * ---------------------------------------------------------------------
 * Copyright (C) 2019 SINOMCU
 * All Rights Reserved
 * ---------------------------------------------------------------------
 * filename: tkm_debug.c
 * verion:   v1.0.0
 * author:   zyx and yj
 * date:     2019/12/11
 * The file is only for SINOMCU MC51F8114 product.
 * ---------------------------------------------------------------------
 */
/********************************************************************
* Includes
********************************************************************/
#include "tkm_debug.h"

/********************************************************************
* Macros and Definitions
********************************************************************/


#if UART_PC_COMMUNICATEFUNCTION == ON
/********************************************************************
* Private variables
********************************************************************/
struct TKM_COMMUNICATE xdata g_tCommunicate;  // debug变量定义

/********************************************************************
* Private functions
********************************************************************/
/***********************************************************
函数名: void tkm_DebugInit(void)
描  述: 触摸库和上位机工具通讯模初始化程序
输入值: 无
输出值: 无
**********************************************************/
void tkm_DebugInit(void)
{
  UART_Init();
  UartPC_CommunicateInit();
}

/***********************************************************
函数名: void UART_Init(void)
描  述: UART初始化子程序
输入值: 无
输出值: 无
**********************************************************/
void UART_Init(void)
{
  P04_OUT_EN; //TX
  P04_FUN_2;  //uart FUN
  P03_IN_EN;  //RX
  P03_OUT_DIS;
  P03_PU_EN; //上拉
  P03_FUN_2; //uart FUN
  UART_TXC &= ~0x04;
  UART_BRL = (uchar)138;
  UART_BRH = (uchar)(138 >> 8);
  UART_TX8_1S;
  UART_RX8;         //UART接收使能
  UART_RX_INT_EN;   //UART接收中断允许
  UART_TXC |= 0x03; //UART发送使能
  UART_TX_INT_DIS;  //禁止UART发送中断，仅在需要发送数据时才使能
  INT3_EN;
}

/*********************************************************
函数名:	void UartPC_CommunicateInit(void)
描  述:	UART与PC端通讯初始化
输入值：无
输出值: 无
**********************************************************/
void UartPC_CommunicateInit(void)
{
  uchar data i;
  g_tCommunicate.uchReceiveDataStatus = 0; // 初始化触摸库通信结构体
  g_tCommunicate.uchrp = 0;
  g_tCommunicate.uchtp = 0;
  g_tCommunicate.uchCorrectDataLength = 0;
  for (i = 0; i < TK_NUM * 2 + 6; i++)
  {
    g_tCommunicate.uchUploadSampByte[i] = 0;
    g_tCommunicate.uchUploadRawByte[i] = 0;
    g_tCommunicate.uchUploadBaseByte[i] = 0;
    g_tCommunicate.uchUploadBuf[i] = 0;
  }
}

/*********************************************************
函数名:	uchar checksum(uchar *prdata, uint length)
描  述:	计算指定的数据校验和
输入值：prdata：计算的数据地址 length：计算的数据长度
输出值: 校验和
**********************************************************/
uchar checksum(uchar *prdata, uint length)
{
	uchar data sum = 0; // Initial value

	while (length--)
	{
		sum += (*prdata);
		prdata++;
	}
	return sum;
}

/*********************************************************
函数名:	void GetUploadData(void)
描  述:	获取向PC端上传的数据
输入值：无
输出值: 无
**********************************************************/
void GetUploadData(void)
{
  uchar data i;
  for (i = 0; i < TK_NUM; i++) // 发送TK数据
  {
    g_tCommunicate.uchUploadSampByte[2 * i] = (uchar)(TK_value_origin[i] >> 8);
    g_tCommunicate.uchUploadSampByte[(2 * i) + 1] = (uchar)(TK_value_origin[i]);
    g_tCommunicate.uchUploadRawByte[2 * i] = (uchar)(TK_value_filter[i] >> 8);
    g_tCommunicate.uchUploadRawByte[(2 * i) + 1] = (uchar)(TK_value_filter[i]);
    g_tCommunicate.uchUploadBaseByte[2 * i] = (uchar)(TK_value_average[i] >> 8);
    g_tCommunicate.uchUploadBaseByte[(2 * i) + 1] = (uchar)(TK_value_average[i]);
  }
}

/*********************************************************
函数名:	void ReceivePCData(void)
描  述:	接收PC端发送数据
输入值：无
输出值: 无
**********************************************************/
void ReceivePCData(void)
{
	uchar data uchReceiveData;
	if (RX_IF) //UART接收中断
	{
		uchReceiveData = UART_RXB; // 读UART接收数据寄存器，同时硬件清除RXIF标志
		if ((UART_RXC & 0xc0) == 0x00)
		{
			if (g_tCommunicate.uchtp == 0) // Not in busy send state
			{
				g_tCommunicate.uchReceiveBuf[g_tCommunicate.uchlength++] = uchReceiveData;
				if (!(g_tCommunicate.uchReceiveDataStatus & 0x01))
				{ // 接收帧数据头
					if (uchReceiveData == 0x69)
					{
						g_tCommunicate.uchReceiveDataStatus |= 0x01;
					}
				}
				else if (!(g_tCommunicate.uchReceiveDataStatus & 0x02))
				{ // 接收Request
					if (uchReceiveData & 0x80)
					{ // 确定是上位机下行数据,接收Request
						g_tCommunicate.uchReceiveDataStatus |= 0x02;
					}
					else
					{
						g_tCommunicate.uchReceiveDataStatus |= 0x40; // 接收数据异常
					}
				}
				else if (!(g_tCommunicate.uchReceiveDataStatus & 0x04))
				{ // 开始接收上位机发送的数据长度
					g_tCommunicate.uchReceiveDataStatus |= 0x04;
					if (uchReceiveData == 0)
					{
						g_tCommunicate.uchReceiveDataStatus |= 0x08;
						g_tCommunicate.uchCorrectDataLength = 1; // 要接收1位校验码
					}
				}
				else if (!(g_tCommunicate.uchReceiveDataStatus & 0x08))
				{ // 开始接收上位机发送的数据内容
				}
				else if (!(g_tCommunicate.uchReceiveDataStatus & 0x10))
				{ // 对上位机发送的数据进行checksum校验
					if (g_tCommunicate.uchCorrectDataLength-- <= 1)
					{ // 接收ckecksum校验码完毕
						if (checksum(&g_tCommunicate.uchReceiveBuf[1], 2 + g_tCommunicate.uchReceiveBuf[2]) == g_tCommunicate.uchReceiveBuf[g_tCommunicate.uchlength - 1])
						{
							g_tCommunicate.uchReceiveDataStatus |= 0x10;
						}
						else
						{
							g_tCommunicate.uchReceiveDataStatus |= 0x40;
						}
					}
				}

				if (g_tCommunicate.uchReceiveDataStatus & 0x40)
				{ // 接收数据异常
					g_tCommunicate.uchReceiveDataStatus = 0;
					g_tCommunicate.uchCorrectDataLength = 0;
					g_tCommunicate.uchlength = 0;
				}
			}
		}
		else
		{
			UART_RXC &= 0xfc;
			UART_RXC |= 0x01;
		}
	}
}

/*********************************************************
函数名:	void SendPCData(void)
描  述:	通过UART向PC端发送数据
输入值：无
输出值: 无
**********************************************************/
void SendPCData(void)
{
	uchar data uchReceiveData;
	uchar data untempdata;
	if ((g_tCommunicate.uchReceiveDataStatus&0x10) && (g_tCommunicate.uchtp == 0))
	{ // 准备发送数据
		g_tCommunicate.uchrp = 0; // 发送指针清零
		g_tCommunicate.uchlength = 0;
		/* 传送帧数据头 */
		g_tCommunicate.uchUploadBuf[g_tCommunicate.uchlength++] = 0x69;
		/* 传送Request命令 */
		g_tCommunicate.uchUploadBuf[g_tCommunicate.uchlength++] = g_tCommunicate.uchReceiveBuf[1] & 0x7f;
		switch (g_tCommunicate.uchUploadBuf[1])
		{
		case 1: // 通道总数
			g_tCommunicate.uchUploadBuf[g_tCommunicate.uchlength++] = 1;
			g_tCommunicate.uchUploadBuf[g_tCommunicate.uchlength++] = TK_NUM;
			untempdata = checksum(&g_tCommunicate.uchUploadBuf[1], 3);
			g_tCommunicate.uchUploadBuf[g_tCommunicate.uchlength] = untempdata;
			break;
		case 2: // 采样数据
			g_tCommunicate.uchUploadBuf[g_tCommunicate.uchlength++] = TK_NUM * 2;
			for (uchReceiveData = 0; uchReceiveData < TK_NUM * 2; uchReceiveData++)
			{
				g_tCommunicate.uchUploadBuf[g_tCommunicate.uchlength++] = g_tCommunicate.uchUploadSampByte[uchReceiveData];
			}
			untempdata = checksum(&g_tCommunicate.uchUploadBuf[1], 2 + TK_NUM * 2);
			g_tCommunicate.uchUploadBuf[g_tCommunicate.uchlength] = untempdata;
			break;
		case 3:
			g_tCommunicate.uchUploadBuf[g_tCommunicate.uchlength++] = TK_NUM * 2;
			for (uchReceiveData = 0; uchReceiveData < TK_NUM * 2; uchReceiveData++)
			{
				g_tCommunicate.uchUploadBuf[g_tCommunicate.uchlength++] = g_tCommunicate.uchUploadRawByte[uchReceiveData];
			}
			untempdata = checksum(&g_tCommunicate.uchUploadBuf[1], 2 + TK_NUM * 2);
			g_tCommunicate.uchUploadBuf[g_tCommunicate.uchlength] = untempdata;
			break;
		case 4:
			g_tCommunicate.uchUploadBuf[g_tCommunicate.uchlength++] = TK_NUM * 2;
			for (uchReceiveData = 0; uchReceiveData < TK_NUM * 2; uchReceiveData++)
			{
				g_tCommunicate.uchUploadBuf[g_tCommunicate.uchlength++] = g_tCommunicate.uchUploadBaseByte[uchReceiveData];
			}
			untempdata = checksum(&g_tCommunicate.uchUploadBuf[1], 2 + TK_NUM * 2);
			g_tCommunicate.uchUploadBuf[g_tCommunicate.uchlength] = untempdata;
			break;
		case 5:
			break;
		default:
			break;
		}
		g_tCommunicate.uchReceiveDataStatus = 0;
		g_tCommunicate.uchCorrectDataLength = 0;
		UART_TXB = g_tCommunicate.uchUploadBuf[0]; //写发送缓冲器
		g_tCommunicate.uchtp = 1;
		UART_TX_INT_EN; //UART 发送中断使能,可触发 INT3 中断
	}
	
	if ( (UARTn_IE & 0x02) &&  TX_IF) //UART发送中断
	{
		if (g_tCommunicate.uchtp <= g_tCommunicate.uchlength)
		{
			UART_TXB = g_tCommunicate.uchUploadBuf[g_tCommunicate.uchtp]; //写发送缓冲器
			g_tCommunicate.uchtp++;
		}
		else
		{
			g_tCommunicate.uchtp = 0; //发送指针清0
			g_tCommunicate.uchlength = 0;
			UART_TX_INT_DIS; //禁止TX中断
		}
	}
}
#endif